//
// $Id: fmsystem.cc,v 1.22 2001/07/13 15:15:06 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//





#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS bx_fmsystem.


bx_fmsystem_c bx_fmsystem;
#if BX_USE_FM_SMF
#define this (&bx_fmsystem)
#endif


bx_fmsystem_c::bx_fmsystem_c(void)
{
}

bx_fmsystem_c::~bx_fmsystem_c(void)
{
  // nothing for now
}


static inline void setbits(Bit8u *addr,int off,unsigned data,int len)
{
  Bit8u *a;
  int b;
  int i;
  int i2;

  a=addr+(off/8);
  b=off%8;
  i2=len-1;

  for(i=0;i<len;i++) {
    *a=(*a&(~(1<<b))) | (((data>>i2)&1)<<b);
    b++;
    if(b==8) {
      a++;
      b=0;
    }
    i2--;
  }
}

  void
bx_fmsystem_c::init(bx_devices_c *d)
{
  // FM-TOWNS SYSTEM I/O

  BX_FMSYSTEM_THIS devices = d;

  BX_FMSYSTEM_THIS soft_reseted=0;

  int i;

  Bit8u fujitsu[]="FUJITSU";
  Bit8u serial[9]={0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8};

  static struct {
    Bit16u num;
    Boolean read;
    Boolean write;
  } fmsystem_ports[]={
    {0x0020,1,1},
    {0x0022,0,1},
    {0x0025,1,0},
    {0x0026,1,0},
    {0x0028,1,1},
    {0x0030,1,0},
    {0x0031,1,0},
    {0x0032,1,1},
    {0x006c,1,1},
    {0x0480,1,1},
    {0x05e8,1,0},
    {0x05ec,1,1},
    {0,0,0}
  };

  i=0;
  while(fmsystem_ports[i].num!=0) {
    if(fmsystem_ports[i].read) {
      BX_FMSYSTEM_THIS devices->register_io_read_handler(this, read_handler,
							 fmsystem_ports[i].num,
							 "FM-TOWNS SYSTEM");
    }
    if(fmsystem_ports[i].write) {
      BX_FMSYSTEM_THIS devices->register_io_write_handler(this, write_handler,
							  fmsystem_ports[i].num,
							  "FM-TOWNS SYSTEM");
    }
    i++;
  }

  memset(BX_FMSYSTEM_THIS serial_rom.data,0,SERIAL_ROM_LEN/8);
  for(i=72;i<=251;i++) {
    setbits(BX_FMSYSTEM_THIS serial_rom.data,i,1,1);
  }
  for(i=0;i<7;i++) {
    setbits(BX_FMSYSTEM_THIS serial_rom.data,251-7-i*8,fujitsu[i],8);
  }
  setbits(BX_FMSYSTEM_THIS serial_rom.data,56,FM_MACHINE_ID,16);
  for(i=0;i<9;i++) {
    setbits(BX_FMSYSTEM_THIS serial_rom.data,55-3-i*4,serial[i],4);
  }
  if (bx_dbg.fmsystem) {
    BX_INFO(("fmsystem: serial ROM =\n"));
    for(i=0;i<SERIAL_ROM_LEN;i+=8) {
      BX_INFO(("         [%.2x]\n",BX_FMSYSTEM_THIS serial_rom.data[i/8]));
    }
  }
}
  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_fmsystem_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_FM_SMF
  bx_fmsystem_c *class_ptr = (bx_fmsystem_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}

  Bit32u
bx_fmsystem_c::read(Bit32u address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif
  Bit8u ret8;
  Bit32u ret = 0;
#define RETURN(x) do { ret = (x); goto read_return; } while (0)

  if(address == 0x0026) {
    if (io_len != 2)
      BX_PANIC(("fmsystem: io read from address %08x len=%u\n",
		(unsigned) address, (unsigned) io_len));
  } else if(address == 0x0030) {
    if (io_len > 2)
      BX_PANIC(("fmsystem: io read from address %08x len=%u\n",
		(unsigned) address, (unsigned) io_len));
  } else {
    if (io_len > 1)
      BX_PANIC(("fmsystem: io read from address %08x len=%u\n",
		(unsigned) address, (unsigned) io_len));
  }

  switch (address) {
    case 0x0020:
      ret8 = BX_FMSYSTEM_THIS soft_reseted ? 1 : 0;
      BX_FMSYSTEM_THIS soft_reseted=0;
      RETURN(ret8);

    case 0x0025:
      RETURN(0x00);

    case 0x0026:
      /* freerun timer ??? */
      BX_FMSYSTEM_THIS freerun_timer-=0x13;
      RETURN(BX_FMSYSTEM_THIS freerun_timer);

    case 0x0028:
      RETURN(BX_FMSYSTEM_THIS NMI_masked ? 0x01 : 0x00);

    case 0x0032:
      RETURN(((BX_FMSYSTEM_THIS serial_rom.data
	       [BX_FMSYSTEM_THIS serial_rom.addr/8] &
	       (1 << (BX_FMSYSTEM_THIS serial_rom.addr % 8))) != 0 ? 1 : 0) |
	     (BX_FMSYSTEM_THIS serial_rom.CLK   ? 0x40 : 0) |
	     (BX_FMSYSTEM_THIS serial_rom.RESET ? 0x80 : 0));

      /* FM-TOWNS i386 */
    case 0x0030:
      if(io_len==2) {
	RETURN(FM_MACHINE_ID);
      } else {
	RETURN(FM_MACHINE_ID & 0xff);
      }
    case 0x0031:
      RETURN((FM_MACHINE_ID >> 8) & 0xff);

    case 0x006c:
      RETURN(0x00);

    case 0x0480:
      RETURN(BX_FMSYSTEM_THIS devices->mem->RAM ? 0x02 : 0);

    case 0x05e8:
      RETURN(BX_FMSYSTEM_THIS devices->mem->megabytes);

    case 0x05ec:
      RETURN(bx_pc_system.get_compatible_mode() ? 0x00 : 0x01);

    default:
      BX_PANIC(("unsupported fmsystem read, address=0x%.4x!\n",
        (unsigned) address));
      RETURN(0);
      break;
    }
  read_return:
  if (bx_dbg.fmsystem)
    BX_INFO(("FMSYSTEM read from address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) ret));
  return ret;
}
#undef RETURN


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_fmsystem_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_FM_SMF
  bx_fmsystem_c *class_ptr = (bx_fmsystem_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}

  void
bx_fmsystem_c::write(Bit32u address, Bit32u value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_FM_SMF
  Boolean newCLK,newRESET;

  if(address == 0x0480) {
  } else {
    if (io_len > 1)
      BX_PANIC(("fmsystem: io write to address %08x len=%u\n",
		(unsigned) address, (unsigned) io_len));
  }

  if (bx_dbg.fmsystem)
    BX_INFO(("FMSYSTEM write to address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) value));

  switch (address) {
    case 0x0020:
      if((value&0x80) != 0) {
	if(!BX_FMSYSTEM_THIS NMI_vector_protect) {
	  BX_INFO(("fmsystem: NMI vector write protected (not yet!!).\n"));
	  BX_FMSYSTEM_THIS NMI_vector_protect=1;
	}
      } else {
	  BX_FMSYSTEM_THIS NMI_vector_protect=0;
      }
      if((value&0x40) != 0) {
	BX_PANIC(("fmsystem: POWER OFF!!\n"));
      }
      if((value&0x01) != 0) {
	BX_PANIC(("fmsystem: software reset is not implemented.\n"));
      }
      return;

    case 0x0022:
      if((value&0x40) != 0) {
	BX_PANIC(("fmsystem: POWER OFF!!\n"));
      }
      return;

    case 0x0028:
      BX_FMSYSTEM_THIS NMI_masked = (value & 0x01) != 0;
      return;

    case 0x0032:
      if((value & 0x20) != 0) {
	BX_PANIC(("fmsystem: CS ID = 1.\n"));
      }
      newCLK   = ((value & 0x40) != 0);
      newRESET = ((value & 0x80) != 0);
      if(BX_FMSYSTEM_THIS serial_rom.CLK == 0 &&
	 newCLK               == 1) {
	BX_FMSYSTEM_THIS serial_rom.addr =
	  (BX_FMSYSTEM_THIS serial_rom.addr+1) % SERIAL_ROM_LEN;
      }
      if(newRESET == 1) {
	BX_FMSYSTEM_THIS serial_rom.addr = 0;
      }
      BX_FMSYSTEM_THIS serial_rom.CLK   = newCLK;
      BX_FMSYSTEM_THIS serial_rom.RESET = newRESET;
      return;

    case 0x006c:
      //usleep(1);
      BX_FMSYSTEM_THIS freerun_timer-=0x54;
      return;

    case 0x0480:
      switch(value & 3) {
      case 0:
      case 1:
      case 2:
      case 3:
	BX_FMSYSTEM_THIS devices->mem->RAM = ((value & 0x02)!=0);
	return;
      default:
	BX_PANIC(("unsupported fmsystem write, address=0x%.4x, value=0x%.2x!\n",
		  (unsigned) address, value));
      }

    case 0x05ec:
      bx_pc_system.set_compatible_mode((value & 0x01) == 0);
      return;

    default:
      BX_PANIC(("unsupported fmsystem write, address=0x%.4x!\n",
        (unsigned) address));
      return;
      break;
    }
}
#endif
